import { GridSignature } from "../../../constants/signature.js";
import { GRID_ROOT_GROUP_ID } from "../rows/gridRowsUtils.js";
import { gridFilteredRowsLookupSelector } from "../filter/gridFilterSelector.js";
import { gridSortedRowIdsSelector } from "../sorting/gridSortingSelector.js";
import { gridRowSelectionManagerSelector } from "./gridRowSelectionSelector.js";
import { gridRowTreeSelector } from "../rows/gridRowsSelector.js";
import { createSelector } from "../../../utils/createSelector.js";
export const ROW_SELECTION_PROPAGATION_DEFAULT = {
  parents: true,
  descendants: true
};
function getGridRowGroupSelectableDescendants(apiRef, groupId) {
  const rowTree = gridRowTreeSelector(apiRef);
  const sortedRowIds = gridSortedRowIdsSelector(apiRef);
  const filteredRowsLookup = gridFilteredRowsLookupSelector(apiRef);
  const groupNode = rowTree[groupId];
  if (!groupNode || groupNode.type !== 'group') {
    return [];
  }
  const descendants = [];
  const startIndex = sortedRowIds.findIndex(id => id === groupId) + 1;
  for (let index = startIndex; index < sortedRowIds.length && rowTree[sortedRowIds[index]]?.depth > groupNode.depth; index += 1) {
    const id = sortedRowIds[index];
    if (filteredRowsLookup[id] !== false && apiRef.current.isRowSelectable(id)) {
      descendants.push(id);
    }
  }
  return descendants;
}
export const checkboxPropsSelector = createSelector(gridRowTreeSelector, gridFilteredRowsLookupSelector, gridRowSelectionManagerSelector, (rowTree, filteredRowsLookup, rowSelectionManager, {
  groupId,
  autoSelectParents
}) => {
  const groupNode = rowTree[groupId];
  if (!groupNode || groupNode.type !== 'group' || rowSelectionManager.has(groupId)) {
    return {
      isIndeterminate: false,
      isChecked: rowSelectionManager.has(groupId)
    };
  }
  let hasSelectedDescendant = false;
  let hasUnSelectedDescendant = false;
  const traverseDescendants = itemToTraverseId => {
    if (filteredRowsLookup[itemToTraverseId] === false ||
    // Perf: Skip checking the rest of the descendants if we already
    // know that there is a selected and an unselected descendant
    hasSelectedDescendant && hasUnSelectedDescendant) {
      return;
    }
    const node = rowTree[itemToTraverseId];
    if (node?.type === 'group') {
      node.children.forEach(traverseDescendants);
    }
    if (rowSelectionManager.has(itemToTraverseId)) {
      hasSelectedDescendant = true;
    } else {
      hasUnSelectedDescendant = true;
    }
  };
  traverseDescendants(groupId);
  return {
    isIndeterminate: hasSelectedDescendant && hasUnSelectedDescendant,
    isChecked: autoSelectParents ? hasSelectedDescendant && !hasUnSelectedDescendant : false
  };
});
export function isMultipleRowSelectionEnabled(props) {
  if (props.signature === GridSignature.DataGrid) {
    // DataGrid Community has multiple row selection enabled only if checkbox selection is enabled.
    return props.checkboxSelection && props.disableMultipleRowSelection !== true;
  }
  return !props.disableMultipleRowSelection;
}
const getRowNodeParents = (tree, id) => {
  const parents = [];
  let parent = id;
  while (parent != null && parent !== GRID_ROOT_GROUP_ID) {
    const node = tree[parent];
    if (!node) {
      return parents;
    }
    parents.push(parent);
    parent = node.parent;
  }
  return parents;
};
const getFilteredRowNodeSiblings = (tree, filteredRows, id) => {
  const node = tree[id];
  if (!node) {
    return [];
  }
  const parent = node.parent;
  if (parent == null) {
    return [];
  }
  const parentNode = tree[parent];
  return parentNode.children.filter(childId => childId !== id && filteredRows[childId] !== false);
};
export const findRowsToSelect = (apiRef, tree, selectedRow, autoSelectDescendants, autoSelectParents, addRow, rowSelectionManager = gridRowSelectionManagerSelector(apiRef)) => {
  const filteredRows = gridFilteredRowsLookupSelector(apiRef);
  const selectedDescendants = new Set([]);
  if (!autoSelectDescendants && !autoSelectParents || filteredRows[selectedRow] === false) {
    return;
  }
  if (autoSelectDescendants) {
    const rowNode = tree[selectedRow];
    if (rowNode?.type === 'group') {
      const descendants = getGridRowGroupSelectableDescendants(apiRef, selectedRow);
      descendants.forEach(rowId => {
        addRow(rowId);
        selectedDescendants.add(rowId);
      });
    }
  }
  if (autoSelectParents) {
    const checkAllDescendantsSelected = rowId => {
      if (!rowSelectionManager.has(rowId) && !selectedDescendants.has(rowId)) {
        return false;
      }
      const node = tree[rowId];
      if (!node) {
        return false;
      }
      if (node.type !== 'group') {
        return true;
      }
      return node.children.every(checkAllDescendantsSelected);
    };
    const traverseParents = rowId => {
      const siblings = getFilteredRowNodeSiblings(tree, filteredRows, rowId);
      if (siblings.length === 0 || siblings.every(checkAllDescendantsSelected)) {
        const rowNode = tree[rowId];
        const parent = rowNode?.parent;
        if (parent != null && parent !== GRID_ROOT_GROUP_ID && apiRef.current.isRowSelectable(parent)) {
          addRow(parent);
          selectedDescendants.add(parent);
          traverseParents(parent);
        }
      }
    };
    traverseParents(selectedRow);
  }
};
export const findRowsToDeselect = (apiRef, tree, deselectedRow, autoSelectDescendants, autoSelectParents, removeRow) => {
  const rowSelectionManager = gridRowSelectionManagerSelector(apiRef);
  if (!autoSelectParents && !autoSelectDescendants) {
    return;
  }
  if (autoSelectParents) {
    const allParents = getRowNodeParents(tree, deselectedRow);
    allParents.forEach(parent => {
      const isSelected = rowSelectionManager.has(parent);
      if (isSelected) {
        removeRow(parent);
      }
    });
  }
  if (autoSelectDescendants) {
    const rowNode = tree[deselectedRow];
    if (rowNode?.type === 'group') {
      const descendants = getGridRowGroupSelectableDescendants(apiRef, deselectedRow);
      descendants.forEach(descendant => {
        removeRow(descendant);
      });
    }
  }
};